<?php
/*
 * Copyright 2017, Google Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
namespace Google\ApiCore;

use Google\Rpc\Code;

class ApiStatus
{
    const OK = 'OK';
    const CANCELLED = 'CANCELLED';
    const UNKNOWN = 'UNKNOWN';
    const INVALID_ARGUMENT = 'INVALID_ARGUMENT';
    const DEADLINE_EXCEEDED = 'DEADLINE_EXCEEDED';
    const NOT_FOUND = 'NOT_FOUND';
    const ALREADY_EXISTS = 'ALREADY_EXISTS';
    const PERMISSION_DENIED = 'PERMISSION_DENIED';
    const RESOURCE_EXHAUSTED = 'RESOURCE_EXHAUSTED';
    const FAILED_PRECONDITION = 'FAILED_PRECONDITION';
    const ABORTED = 'ABORTED';
    const OUT_OF_RANGE = 'OUT_OF_RANGE';
    const UNIMPLEMENTED = 'UNIMPLEMENTED';
    const INTERNAL = 'INTERNAL';
    const UNAVAILABLE = 'UNAVAILABLE';
    const DATA_LOSS = 'DATA_LOSS';
    const UNAUTHENTICATED = 'UNAUTHENTICATED';

    const UNRECOGNIZED_STATUS = 'UNRECOGNIZED_STATUS';

    private static $apiStatusToCodeMap = [
        ApiStatus::OK => Code::OK,
        ApiStatus::CANCELLED => Code::CANCELLED,
        ApiStatus::UNKNOWN => Code::UNKNOWN,
        ApiStatus::INVALID_ARGUMENT => Code::INVALID_ARGUMENT,
        ApiStatus::DEADLINE_EXCEEDED => Code::DEADLINE_EXCEEDED,
        ApiStatus::NOT_FOUND => Code::NOT_FOUND,
        ApiStatus::ALREADY_EXISTS => Code::ALREADY_EXISTS,
        ApiStatus::PERMISSION_DENIED => Code::PERMISSION_DENIED,
        ApiStatus::RESOURCE_EXHAUSTED => Code::RESOURCE_EXHAUSTED,
        ApiStatus::FAILED_PRECONDITION => Code::FAILED_PRECONDITION,
        ApiStatus::ABORTED => Code::ABORTED,
        ApiStatus::OUT_OF_RANGE => Code::OUT_OF_RANGE,
        ApiStatus::UNIMPLEMENTED => Code::UNIMPLEMENTED,
        ApiStatus::INTERNAL => Code::INTERNAL,
        ApiStatus::UNAVAILABLE => Code::UNAVAILABLE,
        ApiStatus::DATA_LOSS => Code::DATA_LOSS,
        ApiStatus::UNAUTHENTICATED => Code::UNAUTHENTICATED,
    ];
    private static $codeToApiStatusMap = [
        Code::OK => ApiStatus::OK,
        Code::CANCELLED => ApiStatus::CANCELLED,
        Code::UNKNOWN => ApiStatus::UNKNOWN,
        Code::INVALID_ARGUMENT => ApiStatus::INVALID_ARGUMENT,
        Code::DEADLINE_EXCEEDED => ApiStatus::DEADLINE_EXCEEDED,
        Code::NOT_FOUND => ApiStatus::NOT_FOUND,
        Code::ALREADY_EXISTS => ApiStatus::ALREADY_EXISTS,
        Code::PERMISSION_DENIED => ApiStatus::PERMISSION_DENIED,
        Code::RESOURCE_EXHAUSTED => ApiStatus::RESOURCE_EXHAUSTED,
        Code::FAILED_PRECONDITION => ApiStatus::FAILED_PRECONDITION,
        Code::ABORTED => ApiStatus::ABORTED,
        Code::OUT_OF_RANGE => ApiStatus::OUT_OF_RANGE,
        Code::UNIMPLEMENTED => ApiStatus::UNIMPLEMENTED,
        Code::INTERNAL => ApiStatus::INTERNAL,
        Code::UNAVAILABLE => ApiStatus::UNAVAILABLE,
        Code::DATA_LOSS => ApiStatus::DATA_LOSS,
        Code::UNAUTHENTICATED => ApiStatus::UNAUTHENTICATED,
    ];

    /**
     * @param string $status
     * @return bool
     */
    public static function isValidStatus($status)
    {
        return array_key_exists($status, self::$apiStatusToCodeMap);
    }

    /**
     * @param int $code
     * @return string
     */
    public static function statusFromRpcCode($code)
    {
        if (array_key_exists($code, self::$codeToApiStatusMap)) {
            return self::$codeToApiStatusMap[$code];
        }
        return ApiStatus::UNRECOGNIZED_STATUS;
    }
}
